<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main Techbiz Core Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class Techbiz_Extension {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Elementor_Test_Extension The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Elementor_Test_Extension An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'init' ] );

	}


	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}

		// Add Plugin actions
		add_action( 'elementor/widgets/register', [ $this, 'init_widgets' ] );

        // Register widget scripts
		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'widget_scripts' ]);

        // category register
		add_action( 'elementor/elements/categories_registered',[ $this, 'techbiz_elementor_widget_categories' ] );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'techbiz' ),
			'<strong>' . esc_html__( 'Techbiz Core', 'techbiz' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'techbiz' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'techbiz' ),
			'<strong>' . esc_html__( 'Techbiz Core', 'techbiz' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'techbiz' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'techbiz' ),
			'<strong>' . esc_html__( 'Techbiz Core', 'techbiz' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'techbiz' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init_widgets() {

		// Include Widget files
		require_once( TECHBIZ_ADDONS . '/widgets/section-title.php' );
		require_once( TECHBIZ_ADDONS . '/widgets/blog-post.php' );
		require_once( TECHBIZ_ADDONS . '/widgets/button.php' );
		require_once( TECHBIZ_ADDONS . '/widgets/testimonial-slider.php' );
		require_once( TECHBIZ_ADDONS . '/widgets/team-member.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/image.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-counter.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-service-slider.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-faq.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-faq-area.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-service-box.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-image-box.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-progress-bar.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-work-process.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-project-slider.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-pricing-table.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-tab-builder.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-projects-filter.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-about-us.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-event.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-testimonial-area.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-project-filter.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-pricing-box.php' );
	 	require_once( TECHBIZ_ADDONS . '/widgets/techbiz-skill-about.php' );

		// Header Elements
		require_once( TECHBIZ_ADDONS . '/header/techbiz-megamenu.php' );
		require_once( TECHBIZ_ADDONS . '/header/techbiz-search.php' );
		require_once( TECHBIZ_ADDONS . '/header/techbiz-mobile-menu.php' );
		require_once( TECHBIZ_ADDONS . '/header/techbiz-offcanvas.php' );
		require_once( TECHBIZ_ADDONS . '/header/techbiz-language-switcher.php' );
		require_once( TECHBIZ_ADDONS . '/header/techbiz-new-header.php' );

		// Footer Elements
		require_once( TECHBIZ_ADDONS . '/footer-widgets/newsletter-widget.php' );
		require_once( TECHBIZ_ADDONS . '/footer-widgets/techbiz-gallery.php' );

		// Register widget
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Section_Title_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Blog_Post() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Button() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Testimonial_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Team_Member() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Image() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Counter() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Service_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Faq() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Faq_Area() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Service_Box() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Image_Box() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Progress_Bar() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Work_Process() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Project_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Pricing_Table() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Tab_Builder() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Project_Filter() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_About_Us_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Event() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Testimonial_Area() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Project_Filter_Two() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Pricing_Box() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Skill_About() );

		// Header Widget Register
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Megamenu() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Search() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Mobilemenu() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Offcanvas() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Language_Switcher() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_New_Header() );

		// Footer Widget Register
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Newsletter_Widgets() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Techbiz_Gallery_Image() );

	}

    public function widget_scripts() {
        wp_enqueue_script(
            'techbiz-frontend-script',
            TECHBIZ_PLUGDIRURI . 'assets/js/techbiz-frontend.js',
            array('jquery'),
            false,
            true
		);
	}

    function techbiz_elementor_widget_categories( $elements_manager ) {
        $elements_manager->add_category(
            'techbiz',
            [
                'title' => __( 'Techbiz', 'techbiz' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
        $elements_manager->add_category(
            'techbiz_footer_elements',
            [
                'title' => __( 'Techbiz Footer Elements', 'techbiz' ),
                'icon' 	=> 'fa fa-plug',
            ]
		);

		$elements_manager->add_category(
            'techbiz_header_elements',
            [
                'title' => __( 'Techbiz Header Elements', 'techbiz' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );

	}

}

Techbiz_Extension::instance();